#region References

using System;
using System.Collections;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using gov.va.med.vbecs.Common;

using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;

#endregion

namespace gov.va.med.vbecs.DAL 
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>10/13/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///	Data access component providing database read/write 
	///	access to patient data
	///</summary>

	#endregion

	public class Patient
	{

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="647"> 
		///		<ExpectedInput>Parameters</ExpectedInput>
		///		<ExpectedOutput>DataTable of patients matching search criteria</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1917"> 
		///		<ExpectedInput>Invalid PatientSSN</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Searches for and returns patients matching input criteria from VBECS
		/// </summary>
		/// <param name="patientName">Patient name</param>
		/// <param name="patientSsn">Patient SSN</param>
		/// <param name="specimenUid">Specimen Uid</param>
		/// <param name="divisionCode">Division Code</param>
		/// <returns>DataTable of patients matching search criteria</returns>
		public static DataTable Search(string patientName, string patientSsn, string specimenUid, string divisionCode)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.PatientSearch.PatientLastName, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.PatientSearch.PatientSsn, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.PatientSearch.specimenuid, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.PatientSearch.DivisionCode, System.Data.SqlDbType.VarChar)
			};

			prms[0].Value = (patientName.Length == 0) ? null : patientName;
			prms[1].Value = (patientSsn.Length == 0) ? null : patientSsn;			
			prms[2].Value = (specimenUid.Length == 0) ? null : specimenUid;
			prms[3].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.PatientSearch.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8215"> 
		///		<ExpectedInput>Valid boolean vaule</ExpectedInput>
		///		<ExpectedOutput>Empty patient table for insert or update</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8216"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetEmptyPatientTableSchema
		/// </summary>
		/// <param name="isUpdate"></param>
		/// <returns></returns>
		public static DataTable GetEmptyPatientTableSchema(bool isUpdate)
		{
			DataTable dtPatient = new DataTable(TABLE.Patient.TableName);

			dtPatient.Columns.Add(TABLE.Patient.PatientGuid, typeof(System.Guid));
			//CR 2431	changed to long to allow for the database change of the VistaPatientId to Bigint
			dtPatient.Columns.Add(TABLE.Patient.VistaPatientId, typeof(long));
			dtPatient.Columns.Add(TABLE.Patient.PatientSsn, typeof(string));
			dtPatient.Columns.Add(TABLE.Patient.SsnPseudoIndicator, typeof(byte));
			dtPatient.Columns.Add(TABLE.Patient.PatientIcn, typeof(string));
			dtPatient.Columns.Add(TABLE.Patient.PatientLastName, typeof(string));
			dtPatient.Columns.Add(TABLE.Patient.PatientFirstName, typeof(string));
			dtPatient.Columns.Add(TABLE.Patient.PatientMiddleName, typeof(string));
			dtPatient.Columns.Add(TABLE.Patient.PatientDob, typeof(System.DateTime));
			dtPatient.Columns.Add(TABLE.Patient.PatientDobCode, typeof(string));
			dtPatient.Columns.Add(TABLE.Patient.PatientDeathDate, typeof(System.DateTime));
			dtPatient.Columns.Add(TABLE.Patient.PatientSexCode, typeof(string));
			dtPatient.Columns.Add(TABLE.Patient.AboRhChangeIndicator, typeof(bool));
			dtPatient.Columns.Add(TABLE.Patient.RecordStatusCode, typeof(string));

			if (isUpdate)
			{
				dtPatient.Columns.Add(TABLE.Patient.RowVersion, typeof(byte[]));
			}

			return(dtPatient);

		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8213"> 
		///		<ExpectedInput>Valid boolean value</ExpectedInput>
		///		<ExpectedOutput>Empty patient treatment table ready for insert or update</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8214"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetEmptyPatientTreatmentTableSchema
		/// </summary>
		/// <param name="isUpdate"></param>
		/// <returns></returns>
		public static DataTable GetEmptyPatientTreatmentTableSchema(bool isUpdate)
		{
			DataTable dtPatientTreatment = new DataTable(TABLE.PatientTreatment.TableName);

			dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.PatientTreatmentGuid, typeof(System.Guid));
			dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.PatientGuid, typeof(System.Guid));
			dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.PatientLocation, typeof(string));
			dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.PatientRoomBed, typeof(string));
			dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.PatientAdmittingDiagnosis, typeof(string));
			dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.InPatientIndicator, typeof(bool));

			if (isUpdate)
			{
				dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.RowVersion, typeof(byte[]));
			}

			return(dtPatientTreatment);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1717"> 
		///		<ExpectedInput>PatientGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of patients details</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1718"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>		
		/// <summary>
		/// Retrieves patient details for a given patient
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static DataTable GetPatientDetails(Guid patientGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientDetails.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetPatientDetails.divisioncode, System.Data.SqlDbType.Char),
			};
			
			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetPatientDetails.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8217"> 
		///		<ExpectedInput>Valid vista patient id and patient icn</ExpectedInput>
		///		<ExpectedOutput>Data table containing patient details</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8218"> 
		///		<ExpectedInput>Invalid patient id and patient icn</ExpectedInput>
		///		<ExpectedOutput>Empty data table returned</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks against the Patient data for existing Patient information (created to remove logic from spInsertPatient)
		/// </summary>
		/// <param name="patientIcn"></param>
		/// CR 2431	changed to long to allow for the database change of the VistaPatientId to Bigint
		/// <param name="vistaPatientId"></param>
		/// <returns></returns>
		public static DataTable GetPatientDetails(long vistaPatientId, string patientIcn)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientDetails.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetPatientDetails.divisioncode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetPatientDetails.searchfield, System.Data.SqlDbType.Int),
				//CR 2431	changed to BigInt to allow for the database change of the VistaPatientId to Bigint
				new SqlParameter(STOREDPROC.GetPatientDetails.vistapatientid, System.Data.SqlDbType.BigInt),
				new SqlParameter(STOREDPROC.GetPatientDetails.patienticn, System.Data.SqlDbType.VarChar)
			};
			
			prms[0].Value = System.Guid.Empty;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
			prms[2].Value = (patientIcn == string.Empty) ? 1 : 2;
			prms[3].Value = vistaPatientId;
			prms[4].Value = patientIcn;

			return Common.StoredProcedure.GetData(STOREDPROC.GetPatientDetails.StoredProcName, prms).Tables[0];
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1719"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1720"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///<summary>Saves special instructions and transfusion requirements</summary>
		/// <param name="sprocArrayList">List of stored procedures to use</param>
		/// <param name="dtArrayList">ArrayList of DataTables with data</param>
		/// <returns>True on success. Otherwise - false.</returns>
		public static bool SaveSIsAndTRs(System.Collections.ArrayList sprocArrayList, System.Collections.ArrayList dtArrayList)
		{
			int returnValue = new StoredProcedure().TransactionalGetValue(sprocArrayList,dtArrayList);
			return (returnValue == 0);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1721"> 
		///		<ExpectedInput>PatientGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of patients abo history</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4128"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// This sproc returns a DataTable containing all of the complete and valid
		/// ABO/Rh tests.  Complete means that both the ABO and Rh parts have been
		/// done.
		/// </summary>
		/// <param name="patientGuid">patientGUID</param>
		/// <returns>Data table</returns>
		public static DataTable GetPatientAboRHHistory(Guid patientGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientAboRHHistory.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetPatientAboRHHistory.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetPatientAboRHHistory.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5956"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5957"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetPatientABORh
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static DataRow GetPatientABORh(Guid patientGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientAboRh.patientguid, System.Data.SqlDbType.UniqueIdentifier)
			};

			prms[0].Value = patientGuid;

			return Common.StoredProcedure.GetData(STOREDPROC.GetPatientAboRh.StoredProcName, prms).Tables[0].Rows[0];
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4028"> 
		///		<ExpectedInput>Valid patient Guid</ExpectedInput>
		///		<ExpectedOutput>Data table of specimen information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4029"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of all specimens associated with the given patient
		/// </summary>
		/// <param name="patientGuid">Patient unique identifier</param>
		/// <param name="startDate">Start date</param>
		/// <param name="endDate">End date</param>
		/// <returns>Data table of specimen information for patient</returns>
		public static DataTable GetAllSpecimensForPatient(Guid patientGuid, DateTime startDate, DateTime endDate)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetSpecimensForPatient.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetSpecimensForPatient.startdate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetSpecimensForPatient.enddate, System.Data.SqlDbType.DateTime),
				new SqlParameter(STOREDPROC.GetSpecimensForPatient.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = patientGuid;
			prms[1].Value = startDate;
			prms[2].Value = endDate;
			prms[3].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetSpecimensForPatient.StoredProcName, prms).Tables[0];

		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1727"> 
		///		<ExpectedInput>PatientGuid</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1732"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns a count of all specimens associated with the given patient
		/// </summary>
		/// <param name="patientGuid">Patient unique identifier</param>
		/// <returns>Specimen count</returns>
		public static int GetAllSpecimensForPatientCount(Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			
			int count = 0;
			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetSpecimensForPatientCount.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetSpecimensForPatientCount.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetSpecimensForPatientCount.StoredProcName, prms).Tables[0];
			
			foreach(DataRow dr in dt.Rows)
			{
				count += (int)dr["Total"];
			}
			
			return(count);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/20/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4107"> 
		///		<ExpectedInput>PatientGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4108"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the ABO/Rh for patient based on the latest ABO/Rh testing
		/// </summary>
		/// <param name="patientGuid">Patient unique identifier</param>
		/// <returns>Data table of recent (last two tests) ABO/Rh information for patient</returns>
		public static DataTable GetRecentPatientAboRHHistory(Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetRecentPatientAboRhHistory.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetRecentPatientAboRhHistory.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetRecentPatientAboRhHistory.StoredProcName, prms);
			
			// There is a possibility that more than 1 table will be returned.  I'm combining them for 
			// easier processing.
			DataTable dtNew = ds.Tables[0].Clone();
			if (ds.Tables.Count > 0)
			{
				foreach (DataTable dt in ds.Tables)
				{
					foreach (DataRow dr in dt.Rows)
					{
						DataRow drNew = dtNew.NewRow();
						drNew.ItemArray = dr.ItemArray;
						dtNew.Rows.Add(drNew);
					}
				}
			}
			return dtNew;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/14/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4109"> 
		///		<ExpectedInput>DataTables</ExpectedInput>
		///		<ExpectedOutput>boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4110"> 
		///		<ExpectedInput>Invalid DataTables</ExpectedInput>
		///		<ExpectedOutput>ArgumentExceptions</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates Patient and SpecimenTest data, "justifying" the unusual change in these records.
		/// </summary>
		/// <param name="patient">Patient DT:PatientGuid, AboRhChangeIndicator, LastUpdateUser and RowVersion</param>
		/// <param name="aboSpecimenTest">Specimen DT for ABO: SpecimenTestGuid, TestResultId, AboRhChangeIndicator, AboRhChangeJustification, LastUpdateUser and RowVersion</param>
		/// <param name="rhSpecimenTest">Specimen DT for Rh: SpecimenTestGuid, TestResultId, AboRhChangeIndicator, AboRhChangeJustification, LastUpdateUser and RowVersion</param>
		/// <param name="lastUpdateFunctionId"></param>
		/// <param name="dtWorkloadEvent"></param>
		/// <returns>Success indicator</returns>
		public static bool JustifyAboRHChange(DataTable patient, DataTable aboSpecimenTest, DataTable rhSpecimenTest, Common.UpdateFunction lastUpdateFunctionId, DataTable dtWorkloadEvent)
		{
			Common.Utility.RequireNonNullColumns(patient, STOREDPROC.UpdatePatientAboRHChangeIndicator.RequiredParameters);
			Common.Utility.RequireNonNullColumns(aboSpecimenTest, STOREDPROC.UpdateSpecimenTest.RequiredParameters);
			Common.Utility.RequireNonNullColumns(rhSpecimenTest, STOREDPROC.UpdateSpecimenTest.RequiredParameters);
			//
			ArrayList alTables = new ArrayList();
			ArrayList alSprocs = new ArrayList();
			//
			if (patient.Rows.Count > 0)
			{
				alTables.Add(Common.Utility.AppendLastUpdateInformation(patient, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.UpdatePatientAboRHChangeIndicator.StoredProcName);
			}
			//
			if (aboSpecimenTest.Rows.Count > 0)
			{
				alTables.Add(Common.Utility.AppendLastUpdateInformation(aboSpecimenTest, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.UpdateSpecimenTest.StoredProcName);
			}
			//
			if (rhSpecimenTest.Rows.Count > 0)
			{
				alTables.Add(Common.Utility.AppendLastUpdateInformation(rhSpecimenTest, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.UpdateSpecimenTest.StoredProcName);
			}
			if (dtWorkloadEvent.Rows.Count > 0)
			{
				alTables.Add(Common.Utility.AppendLastUpdateInformation(dtWorkloadEvent, lastUpdateFunctionId));
				alSprocs.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
			}
			//
			return (new StoredProcedure().TransactionalGetValue(alSprocs, alTables) == 0);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4129"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2142"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Get the list of treating specialties for division
		/// </summary>
		/// <returns></returns>
		public static DataTable GetTreatingSpecialtyList()
		{	
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetTreatingSpecialtyList.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			
			return Common.StoredProcedure.GetData(STOREDPROC.GetTreatingSpecialtyList.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8219"> 
		///		<ExpectedInput>Valid patient guid and current date</ExpectedInput>
		///		<ExpectedOutput>ABO/Rh test table</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8220"> 
		///		<ExpectedInput>Invalid patient guid</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the history of Abo/Rh testing up to the test date
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="testDateTime"></param>
		/// <returns></returns>
		public static DataTable GetAboRHTestHistory(Guid patientGuid, DateTime testDateTime)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAboRhTestHistory.divisioncode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetAboRhTestHistory.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetAboRhTestHistory.testdate, System.Data.SqlDbType.DateTime)
			};
			
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			prms[1].Value = patientGuid;
			prms[2].Value = testDateTime;
			
			return Common.StoredProcedure.GetData(STOREDPROC.GetAboRhTestHistory.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/24/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8087"> 
		///		<ExpectedInput>patient guid that has a justification associated with it</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8088"> 
		///		<ExpectedInput>patient guid that does not have a justification associated with it</ExpectedInput>
		///		<ExpectedOutput>null</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetPatientAboRhJustificationReason
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static DataRow GetPatientAboRhJustificationReason(Guid patientGuid)
		{

			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientAboRhJustificationReason.patientguid, System.Data.SqlDbType.UniqueIdentifier)
			};
			
			prms[0].Value = patientGuid;
				
			try
			{
				return Common.StoredProcedure.GetSingleDataRow(STOREDPROC.GetPatientAboRhJustificationReason.StoredProcName, prms);
			}
			catch (DataAccessLayerException)
			{
				return null;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4775"> 
		///		<ExpectedInput>Valid patient change guid and patient change status</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4776"> 
		///		<ExpectedInput>Invalid patient change Guid</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public static function updates the status of a patient change record and return success indicator
		/// CR 3164: added activeDivisionsAlertViewed
		/// </summary>
		/// <param name="patientChangeGuid">Patient change unique identifier</param>
		/// <param name="patientChangeStatus">Patient change status enum</param>
		/// <param name="activeDivisionsAlertViewed">List of division codes which the patient has active orders, in which the update(s) have not yet been viewed</param>
		/// <param name="rowVersion">Row version</param>
		/// <param name="lastUpdateFunction">Last update function id</param>
		/// <returns>Success indicator</returns>
		public static bool UpdatePatientChangeStatus(
			Guid patientChangeGuid, Common.PatientChangeStatus patientChangeStatus, string activeDivisionsAlertViewed, byte[] rowVersion, Common.UpdateFunction lastUpdateFunction)
		{
			if ( patientChangeGuid == Guid.Empty ) 
				{ throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient change identifier").ResString); }
			if ( patientChangeStatus == Common.PatientChangeStatus.Unknown ) 
				{ throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient change status code").ResString); }
			if ( rowVersion == null ) 
				{ throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("row version").ResString); }
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.UpdatePatientChangeStatus.PatientChangeGuid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.UpdatePatientChangeStatus.PatientChangeStatusCode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.UpdatePatientChangeStatus.ActiveDivisionsAlertViewed, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.UpdatePatientChangeStatus.LastUpdateUser, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.UpdatePatientChangeStatus.RowVersion, System.Data.SqlDbType.Timestamp),
				new SqlParameter(STOREDPROC.UpdatePatientChangeStatus.LastUpdateFunctionId, System.Data.SqlDbType.Int)
			};
			//
			prms[0].Value = patientChangeGuid;
			prms[1].Value = Common.Utility.GetPatientChangeStatusFromCode(patientChangeStatus);
			if ( activeDivisionsAlertViewed == null || activeDivisionsAlertViewed == string.Empty )
			{
				prms[2].Value = DBNull.Value;
			}
			else
			{
				prms[2].Value = activeDivisionsAlertViewed;
			}
			prms[3].Value = Common.LogonUser.LogonUserName;
			prms[4].Value = rowVersion;
			prms[5].Value = (int)lastUpdateFunction;
			//
			return ( new StoredProcedure().TransactionalGetValue(STOREDPROC.UpdatePatientChangeStatus.StoredProcName, prms) == 0 );
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>09/25/2006</CreationDate>
		///
		///<TestCases>
		///
		///<Case type="0" testid ="8451"> 
		///		<ExpectedInput>Valid DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8452"> 
		///		<ExpectedInput>Invalid DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Argument Exception</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements BR_84.07 : The system notifies users of patient merges 
		/// only if one or both of the merged patients are known to VBECS.
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetPendingPatientMergesDetails(string divisionCode)
		{	
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPendingPatientMergesDetails.divisioncode, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = divisionCode;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetPendingPatientMergesDetails.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/21/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4305"> 
		///		<ExpectedInput>Division, Date</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4306"> 
		///		<ExpectedInput>Invalid Division</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements BR_84.07 : The system notifies users of patient merges 
		/// only if one or both of the merged patients are known to VBECS.
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetPendingPatientMerges(string divisionCode)
		{	
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPendingPatientMerges.divisioncode, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = divisionCode;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetPendingPatientMerges.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/25/2006</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8453"> 
		///		<ExpectedInput>Valid Division</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8454"> 
		///		<ExpectedInput>Invalid Division</ExpectedInput>
		///		<ExpectedOutput>Argument Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements BR_84.06 : The system notifies users of patient updates or 
		/// of the patients death only if the patient has current or pending 
		/// orders in the actors division.
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetRecentlyDeceasedPatientDetails(string divisionCode)
		{
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetRecentlyDeceasedPatientDetails.divisioncode, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = divisionCode;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetRecentlyDeceasedPatientDetails.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/21/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4303"> 
		///		<ExpectedInput>Division, Date</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4304"> 
		///		<ExpectedInput>Invalid Division</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements BR_84.06 : The system notifies users of patient updates or 
		/// of the patients death only if the patient has current or pending 
		/// orders in the actors division.
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetRecentlyDeceasedPatients(string divisionCode)
		{
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetRecentlyDeceasedPatients.divisioncode, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = divisionCode;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetRecentlyDeceasedPatients.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>09/25/2006</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8455"> 
		///		<ExpectedInput>Valid Division</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8456"> 
		///		<ExpectedInput>Invalid Division</ExpectedInput>
		///		<ExpectedOutput>Argument Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements BR_84.06 : The system notifies users of patient updates or 
		/// of the patients death only if the patient has current or pending 
		/// orders in the actors division.
		/// </summary>
		/// <returns>DataTable</returns>
		public static DataTable GetRecentlyUpdatedPatientDetails(string divisionCode)
		{
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}

			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetRecentlyUpdatedPatientDetails.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = divisionCode;
			
			return Common.StoredProcedure.GetData(STOREDPROC.GetRecentlyUpdatedPatientDetails.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/25/2006</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4301"> 
		///		<ExpectedInput>Division, Date</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4302"> 
		///		<ExpectedInput>Invalid Division</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements BR_84.06 : The system notifies users of patient updates or 
		/// of the patients death only if the patient has current or pending 
		/// orders in the actors division.
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetRecentlyUpdatedPatients(string divisionCode)
		{
			if (divisionCode == null || divisionCode.Equals(string.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("divisionCode").ResString);
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetRecentlyUpdatedPatients.divisioncode, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = divisionCode;
			//
			return Common.StoredProcedure.GetData(STOREDPROC.GetRecentlyUpdatedPatients.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Hines OIFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2011</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9170"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9171"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the PatientGuid for a DFN and SSN
		/// </summary>
		public static Guid GetPatientGuidForPatient(long vistaPatientId, string patientSsn)
		{
			Guid patientGuid = Guid.Empty;
			//
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientGuidForPatient.vistapatientid, System.Data.SqlDbType.BigInt),
				new SqlParameter(STOREDPROC.GetPatientGuidForPatient.patientssn, System.Data.SqlDbType.Char)
			};
			//
			prms[0].Value = vistaPatientId;
			//
			if ( patientSsn != null )
			{
				prms[1].Value = patientSsn;
			}
			else
			{
				prms[1].Value = DBNull.Value;
			}
			//
			DataSet dsPatient = Common.StoredProcedure.GetData(STOREDPROC.GetPatientGuidForPatient.StoredProcName, prms);
			//
			if ( dsPatient != null && dsPatient.Tables[0] != null && dsPatient.Tables[0].Rows.Count == 1 )
			{
				patientGuid = (Guid)dsPatient.Tables[0].Rows[0][TABLE.Patient.PatientGuid];
			}
			else if ( dsPatient != null && dsPatient.Tables[0] != null && dsPatient.Tables[0].Rows.Count > 1 )
			{
				throw new Exception( "More than 1 Patient was found matching patientSsn = '" + patientSsn + "' and vistaPatientId = '" + vistaPatientId + "'" );
			}
			//
			return patientGuid;
		}

	}
}